<?php
/*
 * Plugin Name: SVG Logo and Text Effects Visual Composer integration
 * Description: Integrates the SVG Logo and Text Effects plugin with WP-Bakery's Visual Composer.
 * Version: 1.0.1
 * Plugin URI: https://www.dashed-slug.net/svg-logo-and-text-effects-extensions/visual-composer-integration-extension
 * Author: Dashed Slug <info@dashed-slug.net>
 * Author URI: http://dashed-slug.net
 * Text Domain: slate-vc
 * Domain Path: /languages/
 * License: GPLv2 or later
 *
 * @package slate-vc
 * @since 1.0.0
 */

// don't load directly
defined( 'ABSPATH' ) || die( '-1' );

include __DIR__ . '/ds-update.php';

if ( ! class_exists( 'SVG_Logo_and_Text_Effects_VC_Integration' ) ) {

	final class SVG_Logo_and_Text_Effects_VC_Integration {

		private static $_instance = null;

		private static $_errors = array();

		private static $_type_to_vc_type = array(
			'color' => 'colorpicker',
			'string' => 'textfield',
			'integer' => 'textfield',
			'float' => 'textfield',
			'html' => 'textfield'
		);

		private function __clone() {
			// Cloning disabled
		}

		/**
		 * Creates or returns an instance of this class.
		 *
		 * @return SVG_Logo_and_Text_Effects_VC_Integration
		 */
		public static function get_instance() {
			if ( ! ( self::$_instance instanceof self ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		public static function error_handler( $errno, $errstr, $errfile, $errline, $errcontext ) {
			if ( ! ( error_reporting() & $errno ) ) {
				return;
			}

			// append error to show on admin_notices
			self::$_errors[] = array(
				'errno' => $errno,
				'errstr' => $errstr,
				'errfile' => $errfile,
				'errline' => $errline,
				'errcontext' => $errcontext,
				'current_filter' => current_filter() );

			error_log( __CLASS__ . ": errno: $errno, errstr: $errstr, errfile: $errfile, errline: $errline" );
		}

		private function __construct() {
			set_error_handler( array( __CLASS__, 'error_handler' ) );

			add_action( 'admin_notices', array( &$this, 'action_admin_notices' ) );
			add_action( 'admin_enqueue_scripts', array( &$this, 'action_admin_enqueue_scripts' ) );

			// Callback for VC integration
			add_action( 'vc_before_init', array( &$this, 'action_vc_before_init' ) );
			add_action( 'vc_load_default_templates_action', array( &$this, 'action_vc_load_default_templates_action' ) );
			add_filter( 'slate_output', array( &$this, 'filter_vc_font_hack' ), 11, 2 );

			// Callbacks for removing buttons from VC's TinyMCE
			add_filter( 'tiny_mce_before_init', array( &$this, 'filter_tiny_mce_before_init' ) );
			add_action( 'after_wp_tiny_mce', array( &$this, 'action_after_wp_tiny_mce' ) );

			restore_error_handler();
		}

		public function filter_tiny_mce_before_init( $mceInit ) {
			$mceInit['plugins'] .= ',slate';
			return $mceInit;
		}

		public function action_after_wp_tiny_mce() {
			printf(
				'<script type="text/javascript" src="%s"></script>',
				plugins_url( 'assets/scripts/slate-tinymce.js', __FILE__ ) );
		}

		public function action_vc_before_init() {
			set_error_handler( array( __CLASS__, 'error_handler' ) );

			if ( is_plugin_active( 'slate/slate.php' ) &&
				 class_exists( 'SVG_Logo_and_Text_Effects' ) &&
				 defined( 'WPB_VC_VERSION' ) ) {

				$slate		= SVG_Logo_and_Text_Effects::get_instance();
				$shapes		= $slate->get_shapes();
				$fonts		= $slate->get_fonts();
				$filters	= $slate->get_filters();
				$fills		= $slate->get_fills();

				foreach ( array_keys( $shapes ) as $shape_template_name ) {
					$shape_template = $shapes[ $shape_template_name ];

					vc_map(
						array(
							'icon' => 'slate-icon',
							'name' => "\"{$shape_template['name']}\" shape template",
							'base' => "slate_{$shape_template['name']}",
							'description' => $shape_template['description'],
							'content_element' => true,
							'show_settings_on_create' => true,
							'category' => 'SVG Logo and Text Effects',
							'admin_enqueue_css' => plugins_url( 'assets/styles/slate-admin.css', __FILE__ ),
							'params' => array(
								array(
									'type' => 'textarea_html',
									'heading' => __( 'Text content', 'slate-vc' ),
									'param_name' => 'content',
									'value' => 'Enter <i>your</i> text <u>here</u>!',
									'group' => 'text',
									'admin_label' => true,
									'description' => __(
										'Enter your text here. You can use the following simple HTML formatting tags: ',
										'slate-vc' ) . implode( ',', SVG_Logo_and_Text_Effects::get_permitted_html_tags() ) ),
								array(
									'type' => 'dropdown',
									'heading' => __( 'Font Family', 'slate-vc' ),
									'param_name' => '_font_family',
									'value' => array_merge( array( '' ), array_keys( $fonts ) ),
									'group' => 'text',
									'admin_label' => true,
									'description' => __(
										'Choose a font family to use for your text, or clear to use the surrounding text\'s font',
										'slate-vc' ) ),

								array(
									'type' => 'textfield',
									'heading' => __( 'Font Size', 'slate-vc' ),
									'param_name' => '_font_size',
									'value' => '',
									'group' => 'text',
									'description' => __(
										'Enter a font size. You may use any valid SVG units and font sizes, such as "12px", "14pt", "120%", etc',
										'slate-vc' ) ),

								array(
									'type' => 'colorpicker',
									'heading' => __( 'Stroke Color', 'slate-vc' ),
									'param_name' => '_stroke',
									'value' => '',
									'group' => 'text',
									'description' => __( 'Choose a color for the text\'s outline.', 'slate-vc' ) ),

								array(
									'type' => 'textfield',
									'heading' => __( 'Stroke Width', 'slate-vc' ),
									'param_name' => '_stroke_width',
									'value' => '1px',
									'group' => 'text',
									'description' => __(
										'Enter a width in pixels for the text\'s outline. This can be a decimal number, e.g. 0.5',
										'slate-vc' ) ),

								array(
									'type' => 'textfield',
									'heading' => __( 'X-axis skew angle', 'slate-vc' ),
									'param_name' => '_skew_x_deg',
									'value' => '0',
									'group' => 'text',
									'description' => __(
										'Enter an X-axis skew angle in degrees. e.g. "-45" or "20"',
										'slate-vc' ) ) ) ) );

				foreach ( $shape_template['params_defs'] as $param_name => $param_def ) {
					if ( '_' != substr( $param_name, 0, 1 ) ) {
						vc_add_param(
							"slate_{$shape_template['name']}",
							array(
								'param_name' => $param_def['name'],
								'type' => self::$_type_to_vc_type[ $param_def['type'] ],
								'heading' => "{$shape_template_name} param: {$param_def['name']}",
								'value' => $param_def['default'],
								'group' => 'shape',
								'description' => "(Default:{$param_def['default']}) {$param_def['description']}" ) );
					}
				}

				vc_add_param(
					"slate_{$shape_template['name']}",
					array(
						'type' => 'dropdown',
						'heading' => __( 'Text Fill', 'slate-vc' ),
						'param_name' => 'fill',
						'value' => array_merge( array( '' ), array_keys( $fills ) ),
						'group' => 'fill',
						'admin_label' => true,
						'description' => __( 'Choose a pattern for the text\'s fill', 'slate-vc' ) ) );

					foreach ( array_keys( $fills ) as $fill_name ) {
						$fill_template = $fills[ $fill_name ];
						foreach ( $fill_template['params_defs'] as $param_def ) {
							if ( '_' != $param_def['name'][0] ) {
								vc_add_param(
									"slate_{$shape_template['name']}",
									array(
										'param_name' => "fill_{$fill_name}_{$param_def['name']}",
										'type' => self::$_type_to_vc_type[ $param_def['type'] ],
										'heading' => "{$fill_name} fill param: {$param_def['name']}",
										'value' => $param_def['default'],
										'group' => 'fill',
										'dependency' => array( 'element' => 'fill', 'value' => array( $fill_name ) ),
										'description' => $param_def['description'] ) );
							}
						}
					}

					foreach ( array_keys( $filters ) as $filter_name ) {
						$filter_template = $filters[ $filter_name ];

						foreach ( $filter_template['params_defs'] as $param_def ) {
							if ( '_' != $param_def['name'][0] )
								vc_add_param(
									"slate_{$shape_template['name']}",
									array(
										'param_name' => "filter_{$filter_name}_{$param_def['name']}",
										'type' => self::$_type_to_vc_type[ $param_def['type'] ],
										'heading' => "$filter_name filter param: {$param_def['name']}",
										'value' => $param_def['default'],
										'description' => $param_def['description'],
										'group' => "{$filter_name} filter" ) );
						}
					}

				} // end foreach shape template

			} // end if vc and slate plugins loaded

			restore_error_handler();
		}

		public function action_vc_load_default_templates_action() {

			$slate		= SVG_Logo_and_Text_Effects::get_instance();
			$presets	= $slate->get_presets();

			foreach ( $presets as $preset_name => $preset ) {
				$template = array(
					'name' => "SVG text: $preset_name",
					'weight' => 0,
					// Thumbnail should have this dimensions: 114x154px
					'image_path' => str_replace( ' ', '%20',
						plugins_url( 'assets/sprites/logo-114x154.png', __FILE__ ) ),
					'custom_class' => '' );

						$template['content'] = $preset;
						vc_add_default_templates( $template );
			}

		}

		public function action_admin_notices() {
			set_error_handler( array( __CLASS__, 'error_handler' ) );

			if ( ! defined( 'WPB_VC_VERSION' ) ) {
				echo '<div class="notice notice-warn is-dismissible"><p>' . sprintf(
					__( '<strong>%s</strong> requires the <strong><a href="http://www.codecanyon.net/item/visual-composer-page-builder-for-wordpress/242431?ref=dashed-slug" target="_blank">Visual Composer</a></strong> plugin. Please install and activate it on your site.', 'slate-vc' ),
					'SVG Logo and Text Effects Visual Composer integration' ) . '</p></div>';
			}

			if ( ! is_plugin_active( 'slate/slate.php' ) ) {
				echo '<div class="notice notice-warn is-dismissible"><p>' . sprintf(
					__( '<strong>%s</strong> requires the <strong><a href="http://dashed-slug.net/slate" target="_blank">SVG Logo and Text Effects</a></strong> plugin. Please install and activate it on your site.', 'slate-vc' ),
					'SVG Logo and Text Effects Visual Composer integration' ) . '</p></div>';
			}

			foreach ( self::$_errors as $error ) {
				if ( true ) {
					$level = 'notice';
					if ( E_USER_ERROR == $error['errno'] ) {
						$level = 'error';
					} elseif ( E_USER_WARNING == $error['errno'] ) {
						$level = 'warning';
					}

					echo "<div class=\"notice notice-$level is-dismissible\">";
					echo "<p style=\"font-size: larger;\">{$error['errstr']}</p><pre>" . 'SVG Logo and Text Effects Visual Composer integration' .
					" on action <code>{$error['current_filter']}</code>, <code>{$error['errfile']}</code>, line <code>{$error['errline']}</code>.</pre></div>";
				}
			}
			restore_error_handler();
		} // end admin notices

		/**
		 * hack to load fonts dynamically on VC frontend editor without page reload
		 */
		public function filter_vc_font_hack( $output, $vars ) {

			if ( isset( $vars['atts'] ) && isset( $vars['atts']['_font_family'] ) ) {
				$font_family = $vars['atts']['_font_family'];

				if ( isset( $_POST ) && array_key_exists( 'action', $_POST ) && 'vc_load_shortcode' == $_POST['action'] ) {

					$slate		= SVG_Logo_and_Text_Effects::get_instance();
					$fonts		= $slate->get_fonts();
					$font 		= $fonts[ $font_family ];

					if ( isset( $font['source'] ) ) {
						if ( 'google' == $font['source'] ) {

							return $output . '<link href="https://fonts.googleapis.com/css?family=' . $font_family .
							'" rel="stylesheet" type="text/css">';

						} elseif ( 'custom' == $font['source'] ) {

							return "$output<style>@font-face { font-family: '$font_family'; src: url('{$font['url']}'); }</style>";
						}
					} // end if isset font source
				} // end if vc frontend request
			} // end if font family is set
			return $output;
		} // end filter_vc_font_hack

		public function action_admin_enqueue_scripts() {
			set_error_handler( array( __CLASS__, 'error_handler' ) );

			if ( file_exists( __DIR__ . '/assets/styles/slate-vc-admin.min.css' ) ) {
				$slate_vc_admin_style = 'assets/styles/slate-vc-admin.min.css';
			} else {
				$slate_vc_admin_style = 'assets/styles/slate-vc-admin.css';
			}

			wp_enqueue_style( 'slate-admin', plugins_url( $slate_vc_admin_style, __FILE__ ) );

			restore_error_handler();
		}

	} // end class
} // end if class defined

SVG_Logo_and_Text_Effects_VC_Integration::get_instance();
